"use client";

import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { useState } from "react";
import { useRouter } from "next/navigation";
import { toast } from "sonner";
import { Upload, FileText, CheckCircle2, X } from "lucide-react";

export default function VisaApplicationPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [step, setStep] = useState(1);
  const [uploadedFiles, setUploadedFiles] = useState<{[key: string]: File}>({});

  const [formData, setFormData] = useState({
    // Kişisel Bilgiler
    firstName: "",
    lastName: "",
    email: "",
    phone: "",
    birthDate: "",
    birthPlace: "",
    nationality: "",
    passportNo: "",
    passportExpiry: "",

    // Vize Bilgileri
    country: "",
    visaType: "",
    travelDate: "",
    returnDate: "",
    purpose: "",

    // Adres Bilgileri
    address: "",
    city: "",
    postalCode: "",

    // Ek Bilgiler
    notes: "",
  });

  const requiredDocuments = [
    { id: "passport", label: "Pasaport Fotokopisi", required: true },
    { id: "photo", label: "Fotoğraf (5x6 cm)", required: true },
    { id: "application", label: "Başvuru Formu", required: true },
    { id: "financial", label: "Mali Durum Belgesi", required: true },
    { id: "accommodation", label: "Konaklama Belgesi", required: false },
    { id: "insurance", label: "Seyahat Sigortası", required: false },
  ];

  const handleFileUpload = (docId: string, file: File) => {
    setUploadedFiles(prev => ({ ...prev, [docId]: file }));
    toast.success(`${file.name} yüklendi`);
  };

  const removeFile = (docId: string) => {
    const newFiles = { ...uploadedFiles };
    delete newFiles[docId];
    setUploadedFiles(newFiles);
    toast.info("Dosya kaldırıldı");
  };

  // Validation Functions
  const validateStep1 = () => {
    const { firstName, lastName, email, phone, birthDate, birthPlace, nationality, passportNo, passportExpiry } = formData;

    if (!firstName.trim() || firstName.length < 2) {
      toast.error("Geçerli bir ad girin (en az 2 karakter)");
      return false;
    }
    if (!lastName.trim() || lastName.length < 2) {
      toast.error("Geçerli bir soyad girin (en az 2 karakter)");
      return false;
    }
    if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
      toast.error("Geçerli bir e-posta adresi girin");
      return false;
    }
    if (!/^[0-9]{10,}$/.test(phone.replace(/[\s()-]/g, ""))) {
      toast.error("Geçerli bir telefon numarası girin (en az 10 rakam)");
      return false;
    }
    if (!birthDate) {
      toast.error("Doğum tarihi seçin");
      return false;
    }
    const birthYear = new Date(birthDate).getFullYear();
    const currentYear = new Date().getFullYear();
    if (currentYear - birthYear < 18 || currentYear - birthYear > 100) {
      toast.error("Yaşınız 18-100 arasında olmalıdır");
      return false;
    }
    if (!birthPlace.trim()) {
      toast.error("Doğum yeri girin");
      return false;
    }
    if (!nationality.trim()) {
      toast.error("Uyruk bilgisi girin");
      return false;
    }
    if (!passportNo.trim() || passportNo.length < 6) {
      toast.error("Geçerli bir pasaport numarası girin (en az 6 karakter)");
      return false;
    }
    if (!passportExpiry) {
      toast.error("Pasaport son geçerlilik tarihi seçin");
      return false;
    }
    const expiryDate = new Date(passportExpiry);
    const today = new Date();
    const sixMonthsLater = new Date();
    sixMonthsLater.setMonth(sixMonthsLater.getMonth() + 6);
    if (expiryDate < sixMonthsLater) {
      toast.error("Pasaportunuz en az 6 ay geçerli olmalıdır");
      return false;
    }
    return true;
  };

  const validateStep2 = () => {
    const { country, visaType, travelDate, returnDate, purpose, address, city, postalCode } = formData;

    if (!country) {
      toast.error("Lütfen ülke seçin");
      return false;
    }
    if (!visaType) {
      toast.error("Lütfen vize türü seçin");
      return false;
    }
    if (!travelDate) {
      toast.error("Seyahat tarihi seçin");
      return false;
    }
    if (!returnDate) {
      toast.error("Dönüş tarihi seçin");
      return false;
    }
    const travel = new Date(travelDate);
    const returnD = new Date(returnDate);
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    if (travel < today) {
      toast.error("Seyahat tarihi bugünden ileri bir tarih olmalıdır");
      return false;
    }
    if (returnD <= travel) {
      toast.error("Dönüş tarihi, seyahat tarihinden sonra olmalıdır");
      return false;
    }
    if (!purpose.trim() || purpose.length < 20) {
      toast.error("Seyahat amacını detaylı açıklayın (en az 20 karakter)");
      return false;
    }
    if (!address.trim() || address.length < 10) {
      toast.error("Geçerli bir adres girin (en az 10 karakter)");
      return false;
    }
    if (!city.trim() || city.length < 2) {
      toast.error("Şehir bilgisi girin");
      return false;
    }
    if (!/^[0-9]{5}$/.test(postalCode)) {
      toast.error("Geçerli bir posta kodu girin (5 rakam)");
      return false;
    }
    return true;
  };

  const validateStep3 = () => {
    const requiredDocs = requiredDocuments.filter(doc => doc.required);
    const missingDocs = requiredDocs.filter(doc => !uploadedFiles[doc.id]);

    if (missingDocs.length > 0) {
      toast.error(`❌ Zorunlu evraklar eksik: ${missingDocs.map(d => d.label).join(", ")}`);
      return false;
    }

    // Check file sizes (max 5MB)
    for (const [docId, file] of Object.entries(uploadedFiles)) {
      if (file.size > 5 * 1024 * 1024) {
        toast.error(`${file.name} dosyası çok büyük (maksimum 5MB olmalıdır)`);
        return false;
      }
      // Check file type
      const validTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png'];
      if (!validTypes.includes(file.type)) {
        toast.error(`${file.name} geçersiz dosya formatı (sadece PDF, JPG, PNG kabul edilir)`);
        return false;
      }
    }

    return true;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (step === 1) {
      if (!validateStep1()) return;
      setStep(2);
      toast.success("✅ Kişisel bilgiler onaylandı");
      window.scrollTo({ top: 0, behavior: 'smooth' });
      return;
    }

    if (step === 2) {
      if (!validateStep2()) return;
      setStep(3);
      toast.success("✅ Vize bilgileri onaylandı, lütfen evraklarınızı yükleyin");
      window.scrollTo({ top: 0, behavior: 'smooth' });
      return;
    }

    if (!validateStep3()) return;

    setLoading(true);

    // Simulate submission
    setTimeout(() => {
      // Save to localStorage
      const applications = JSON.parse(localStorage.getItem("applications") || "[]");
      const newApplication = {
        id: Date.now(),
        ...formData,
        documents: Object.keys(uploadedFiles).map(key => ({
          id: key,
          name: uploadedFiles[key].name,
          size: uploadedFiles[key].size,
          type: uploadedFiles[key].type,
        })),
        status: "Beklemede",
        statusColor: "yellow",
        date: new Date().toLocaleDateString("tr-TR"),
        createdAt: new Date().toISOString(),
        amount: "2.124₺",
        paymentStatus: "Ödeme Bekliyor",
      };
      applications.push(newApplication);
      localStorage.setItem("applications", JSON.stringify(applications));

      toast.success("✅ Başvurunuz başarıyla alındı! Ödeme sayfasına yönlendiriliyorsunuz...");
      setTimeout(() => {
        router.push("/payment");
      }, 1500);
    }, 1500);
  };

  return (
    <div className="pt-20 min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-12">
        <div className="text-center mb-8">
          <Badge className="mb-4 bg-emerald-100 text-emerald-700">Vize Başvurusu</Badge>
          <h1 className="text-4xl font-bold mb-4">Vize Başvuru Formu</h1>
          <p className="text-gray-600">Formu eksiksiz doldurup evraklarınızı yükleyin</p>
        </div>

        {/* Progress Steps */}
        <div className="flex items-center justify-center mb-12">
          {[1, 2, 3].map((s) => (
            <div key={s} className="flex items-center">
              <div className={`w-10 h-10 rounded-full flex items-center justify-center font-bold ${
                step >= s ? "bg-emerald-600 text-white" : "bg-gray-200 text-gray-600"
              }`}>
                {s}
              </div>
              {s < 3 && (
                <div className={`w-20 h-1 ${step > s ? "bg-emerald-600" : "bg-gray-200"}`} />
              )}
            </div>
          ))}
        </div>

        <form onSubmit={handleSubmit}>
          {/* Step 1: Personal Info */}
          {step === 1 && (
            <Card className="p-8">
              <h2 className="text-2xl font-bold mb-6">1. Kişisel Bilgiler</h2>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <Label htmlFor="firstName">Ad *</Label>
                  <Input
                    id="firstName"
                    value={formData.firstName}
                    onChange={(e) => setFormData({ ...formData, firstName: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="lastName">Soyad *</Label>
                  <Input
                    id="lastName"
                    value={formData.lastName}
                    onChange={(e) => setFormData({ ...formData, lastName: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="email">E-posta *</Label>
                  <Input
                    id="email"
                    type="email"
                    value={formData.email}
                    onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="phone">Telefon *</Label>
                  <Input
                    id="phone"
                    type="tel"
                    value={formData.phone}
                    onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="birthDate">Doğum Tarihi *</Label>
                  <Input
                    id="birthDate"
                    type="date"
                    value={formData.birthDate}
                    onChange={(e) => setFormData({ ...formData, birthDate: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="birthPlace">Doğum Yeri *</Label>
                  <Input
                    id="birthPlace"
                    value={formData.birthPlace}
                    onChange={(e) => setFormData({ ...formData, birthPlace: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="nationality">Uyruk *</Label>
                  <Input
                    id="nationality"
                    value={formData.nationality}
                    onChange={(e) => setFormData({ ...formData, nationality: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="passportNo">Pasaport No *</Label>
                  <Input
                    id="passportNo"
                    value={formData.passportNo}
                    onChange={(e) => setFormData({ ...formData, passportNo: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="passportExpiry">Pasaport Son Geçerlilik *</Label>
                  <Input
                    id="passportExpiry"
                    type="date"
                    value={formData.passportExpiry}
                    onChange={(e) => setFormData({ ...formData, passportExpiry: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
              </div>
              <Button type="submit" className="w-full mt-8 bg-emerald-600 hover:bg-emerald-700 py-6">
                Devam Et
              </Button>
            </Card>
          )}

          {/* Step 2: Visa Info */}
          {step === 2 && (
            <Card className="p-8">
              <h2 className="text-2xl font-bold mb-6">2. Vize Bilgileri</h2>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <Label htmlFor="country">Ülke *</Label>
                  <select
                    id="country"
                    value={formData.country}
                    onChange={(e) => setFormData({ ...formData, country: e.target.value })}
                    required
                    className="mt-2 w-full border rounded-md p-2"
                  >
                    <option value="">Seçiniz</option>
                    <option value="Amerika">Amerika</option>
                    <option value="İngiltere">İngiltere</option>
                    <option value="Kanada">Kanada</option>
                    <option value="Almanya">Almanya</option>
                    <option value="Fransa">Fransa</option>
                  </select>
                </div>
                <div>
                  <Label htmlFor="visaType">Vize Türü *</Label>
                  <select
                    id="visaType"
                    value={formData.visaType}
                    onChange={(e) => setFormData({ ...formData, visaType: e.target.value })}
                    required
                    className="mt-2 w-full border rounded-md p-2"
                  >
                    <option value="">Seçiniz</option>
                    <option value="Turist">Turist Vizesi</option>
                    <option value="Öğrenci">Öğrenci Vizesi</option>
                    <option value="İş">İş Vizesi</option>
                  </select>
                </div>
                <div>
                  <Label htmlFor="travelDate">Seyahat Tarihi *</Label>
                  <Input
                    id="travelDate"
                    type="date"
                    value={formData.travelDate}
                    onChange={(e) => setFormData({ ...formData, travelDate: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="returnDate">Dönüş Tarihi *</Label>
                  <Input
                    id="returnDate"
                    type="date"
                    value={formData.returnDate}
                    onChange={(e) => setFormData({ ...formData, returnDate: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div className="md:col-span-2">
                  <Label htmlFor="purpose">Seyahat Amacı *</Label>
                  <Textarea
                    id="purpose"
                    value={formData.purpose}
                    onChange={(e) => setFormData({ ...formData, purpose: e.target.value })}
                    required
                    className="mt-2"
                    rows={4}
                  />
                </div>
                <div className="md:col-span-2">
                  <Label htmlFor="address">Adres *</Label>
                  <Input
                    id="address"
                    value={formData.address}
                    onChange={(e) => setFormData({ ...formData, address: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="city">Şehir *</Label>
                  <Input
                    id="city"
                    value={formData.city}
                    onChange={(e) => setFormData({ ...formData, city: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
                <div>
                  <Label htmlFor="postalCode">Posta Kodu *</Label>
                  <Input
                    id="postalCode"
                    value={formData.postalCode}
                    onChange={(e) => setFormData({ ...formData, postalCode: e.target.value })}
                    required
                    className="mt-2"
                  />
                </div>
              </div>
              <div className="flex gap-4 mt-8">
                <Button type="button" variant="outline" onClick={() => setStep(1)} className="flex-1 py-6">
                  Geri
                </Button>
                <Button type="submit" className="flex-1 bg-emerald-600 hover:bg-emerald-700 py-6">
                  Devam Et
                </Button>
              </div>
            </Card>
          )}

          {/* Step 3: Documents */}
          {step === 3 && (
            <Card className="p-8">
              <h2 className="text-2xl font-bold mb-6">3. Evrak Yükleme</h2>
              <div className="space-y-4">
                {requiredDocuments.map((doc) => (
                  <Card key={doc.id} className="p-4 border-2">
                    <div className="flex items-center justify-between">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-2">
                          <FileText className="h-5 w-5 text-emerald-600" />
                          <span className="font-medium">{doc.label}</span>
                          {doc.required && <Badge variant="destructive" className="text-xs">Zorunlu</Badge>}
                        </div>
                        {uploadedFiles[doc.id] && (
                          <div className="flex items-center gap-2 text-sm text-green-600">
                            <CheckCircle2 className="h-4 w-4" />
                            <span>{uploadedFiles[doc.id].name}</span>
                          </div>
                        )}
                      </div>
                      <div className="flex gap-2">
                        {uploadedFiles[doc.id] ? (
                          <Button
                            type="button"
                            variant="outline"
                            size="sm"
                            onClick={() => removeFile(doc.id)}
                          >
                            <X className="h-4 w-4" />
                          </Button>
                        ) : (
                          <label className="cursor-pointer">
                            <input
                              type="file"
                              className="hidden"
                              accept=".pdf,.jpg,.jpeg,.png"
                              onChange={(e) => {
                                const file = e.target.files?.[0];
                                if (file) handleFileUpload(doc.id, file);
                              }}
                            />
                            <div className="bg-emerald-600 text-white px-4 py-2 rounded-md hover:bg-emerald-700 flex items-center gap-2">
                              <Upload className="h-4 w-4" />
                              Yükle
                            </div>
                          </label>
                        )}
                      </div>
                    </div>
                  </Card>
                ))}
              </div>

              <div className="mt-8">
                <Label htmlFor="notes">Ek Notlar</Label>
                <Textarea
                  id="notes"
                  value={formData.notes}
                  onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                  className="mt-2"
                  rows={4}
                  placeholder="Varsa belirtmek istediğiniz ek bilgiler..."
                />
              </div>

              <div className="flex gap-4 mt-8">
                <Button type="button" variant="outline" onClick={() => setStep(2)} className="flex-1 py-6">
                  Geri
                </Button>
                <Button type="submit" disabled={loading} className="flex-1 bg-emerald-600 hover:bg-emerald-700 py-6">
                  {loading ? "Gönderiliyor..." : "Başvuruyu Tamamla"}
                </Button>
              </div>
            </Card>
          )}
        </form>
      </div>
    </div>
  );
}
