"use client";

import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Users, FileText, DollarSign, TrendingUp, CheckCircle2, Clock, XCircle, Eye, Download } from "lucide-react";
import { useEffect, useState } from "react";
import { toast } from "sonner";

export default function AdminPage() {
  const [applications, setApplications] = useState<any[]>([]);
  const [enrollments, setEnrollments] = useState<any[]>([]);
  const [selectedApp, setSelectedApp] = useState<any>(null);
  const [stats, setStats] = useState({
    totalUsers: 0,
    totalApplications: 0,
    pending: 0,
    approved: 0,
    totalRevenue: 0,
  });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = () => {
    const apps = JSON.parse(localStorage.getItem("applications") || "[]");
    const enroll = JSON.parse(localStorage.getItem("enrollments") || "[]");

    setApplications(apps);
    setEnrollments(enroll);

    const allApps = [...apps, ...enroll];
    const pending = allApps.filter(a => a.status === "Beklemede").length;
    const approved = allApps.filter(a => a.status === "Onaylandı").length;
    const revenue = allApps.reduce((sum, a) => {
      const amount = parseInt(a.amount?.replace(/[^0-9]/g, "") || "0");
      return sum + amount;
    }, 0);

    setStats({
      totalUsers: new Set(allApps.map(a => a.email)).size,
      totalApplications: allApps.length,
      pending,
      approved,
      totalRevenue: revenue,
    });
  };

  const updateStatus = (id: number, newStatus: string, type: "application" | "enrollment") => {
    if (type === "application") {
      const updated = applications.map(app =>
        app.id === id ? { ...app, status: newStatus } : app
      );
      localStorage.setItem("applications", JSON.stringify(updated));
      setApplications(updated);
    } else {
      const updated = enrollments.map(enroll =>
        enroll.id === id ? { ...enroll, status: newStatus } : enroll
      );
      localStorage.setItem("enrollments", JSON.stringify(updated));
      setEnrollments(updated);
    }
    loadData();
    toast.success(`Durum güncellendi: ${newStatus}`);
  };

  const allApplications = [
    ...applications.map(a => ({ ...a, type: "Vize", typeLabel: `Vize - ${a.country}` })),
    ...enrollments.map(e => ({ ...e, type: "Dil Okulu", typeLabel: `Dil Okulu - ${e.school}` }))
  ].sort((a, b) => new Date(b.createdAt || b.date).getTime() - new Date(a.createdAt || a.date).getTime());
  return (
    <div className="pt-20 min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 py-12">
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">Admin Panel</h1>
          <p className="text-gray-600">Hoş geldiniz, Yönetici</p>
        </div>

        {/* Stats Grid */}
        <div className="grid md:grid-cols-4 gap-6 mb-8">
          <Card className="p-6">
            <div className="flex items-center justify-between mb-4">
              <Users className="h-8 w-8 text-blue-600" />
            </div>
            <div className="text-3xl font-bold mb-1">{stats.totalUsers}</div>
            <div className="text-gray-600">Toplam Kullanıcı</div>
          </Card>

          <Card className="p-6">
            <div className="flex items-center justify-between mb-4">
              <FileText className="h-8 w-8 text-emerald-600" />
            </div>
            <div className="text-3xl font-bold mb-1">{stats.totalApplications}</div>
            <div className="text-gray-600">Toplam Başvuru</div>
          </Card>

          <Card className="p-6">
            <div className="flex items-center justify-between mb-4">
              <Clock className="h-8 w-8 text-yellow-600" />
              {stats.pending > 0 && <Badge className="bg-yellow-100 text-yellow-700">Acil</Badge>}
            </div>
            <div className="text-3xl font-bold mb-1">{stats.pending}</div>
            <div className="text-gray-600">Bekleyen Başvuru</div>
          </Card>

          <Card className="p-6">
            <div className="flex items-center justify-between mb-4">
              <DollarSign className="h-8 w-8 text-green-600" />
            </div>
            <div className="text-3xl font-bold mb-1">₺{(stats.totalRevenue / 1000).toFixed(0)}K</div>
            <div className="text-gray-600">Toplam Gelir</div>
          </Card>
        </div>

        {/* Quick Actions */}
        <Card className="p-6 mb-8">
          <h2 className="text-xl font-bold mb-4">Hızlı İşlemler</h2>
          <div className="flex flex-wrap gap-3">
            <Button className="bg-emerald-600 hover:bg-emerald-700">Yeni Başvuru Ekle</Button>
            <Button variant="outline">Dil Okulu Ekle</Button>
            <Button variant="outline">Vize Türü Ekle</Button>
            <Button variant="outline">Kullanıcı Ekle</Button>
            <Button variant="outline">Rapor Oluştur</Button>
          </div>
        </Card>

        <div className="grid md:grid-cols-3 gap-8">
          {/* Recent Applications */}
          <Card className="md:col-span-2 p-6">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-2xl font-bold">Son Başvurular</h2>
              <Button variant="outline" size="sm">Tümünü Gör</Button>
            </div>

            <div className="space-y-4">
              {allApplications.length === 0 ? (
                <Card className="p-8 text-center">
                  <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-gray-600">Henüz başvuru bulunmuyor</p>
                  <p className="text-sm text-gray-500 mt-2">Yeni başvurular burada görünecektir</p>
                </Card>
              ) : (
                allApplications.slice(0, 10).map((app) => (
                  <Card key={app.id} className="p-4 hover:shadow-md transition-shadow">
                    <div className="flex items-center justify-between">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <span className="font-mono text-sm text-gray-500">#{app.id}</span>
                          <Badge className={`${
                            app.type === "Vize" ? "bg-emerald-100 text-emerald-700" : "bg-blue-100 text-blue-700"
                          }`}>
                            {app.type}
                          </Badge>
                          <Badge variant={
                            app.status === "Onaylandı" ? "default" :
                            app.status === "Beklemede" ? "secondary" :
                            "destructive"
                          }>
                            {app.status === "Onaylandı" && <CheckCircle2 className="h-3 w-3 mr-1" />}
                            {app.status === "Beklemede" && <Clock className="h-3 w-3 mr-1" />}
                            {app.status === "Reddedildi" && <XCircle className="h-3 w-3 mr-1" />}
                            {app.status}
                          </Badge>
                        </div>
                        <h3 className="font-bold">{app.firstName} {app.lastName || app.name}</h3>
                        <p className="text-sm text-gray-600">{app.typeLabel}</p>
                        <p className="text-xs text-gray-500 mt-1">{app.email}</p>
                      </div>
                      <div className="text-right">
                        <div className="font-bold text-emerald-600 mb-3">{app.amount}</div>
                        <div className="flex gap-2">
                          <Button size="sm" variant="outline" onClick={() => setSelectedApp(app)}>
                            <Eye className="h-4 w-4 mr-1" />
                            Detay
                          </Button>
                          {app.status === "Beklemede" && (
                            <Button
                              size="sm"
                              className="bg-green-600 hover:bg-green-700"
                              onClick={() => updateStatus(app.id, "Onaylandı", app.type === "Vize" ? "application" : "enrollment")}
                            >
                              Onayla
                            </Button>
                          )}
                        </div>
                      </div>
                    </div>
                  </Card>
                ))
              )}
            </div>
          </Card>

          {/* Activity Feed */}
          <Card className="p-6">
            <h2 className="text-xl font-bold mb-6">Son Aktiviteler</h2>
            <div className="space-y-4">
              {allApplications.slice(0, 5).length === 0 ? (
                <p className="text-gray-500 text-sm text-center py-4">Henüz aktivite yok</p>
              ) : (
                allApplications.slice(0, 5).map((activity, idx) => {
                  const timeAgo = () => {
                    const date = new Date(activity.createdAt || activity.date);
                    const now = new Date();
                    const diff = Math.floor((now.getTime() - date.getTime()) / 1000 / 60);
                    if (diff < 1) return "Az önce";
                    if (diff < 60) return `${diff} dk önce`;
                    if (diff < 1440) return `${Math.floor(diff / 60)} sa önce`;
                    return `${Math.floor(diff / 1440)} gün önce`;
                  };

                  return (
                    <div key={idx} className="flex items-start gap-3 pb-3 border-b last:border-0">
                      <div className={`w-2 h-2 rounded-full mt-2 ${
                        activity.type === "Vize" ? "bg-emerald-600" : "bg-blue-600"
                      }`} />
                      <div className="flex-1">
                        <p className="font-medium text-sm">
                          {activity.type === "Vize" ? "Yeni vize başvurusu" : "Yeni dil okulu kaydı"}
                        </p>
                        <p className="text-xs text-gray-600">{activity.firstName} {activity.lastName || activity.name}</p>
                      </div>
                      <span className="text-xs text-gray-500">{timeAgo()}</span>
                    </div>
                  );
                })
              )}
            </div>
          </Card>
        </div>
      </div>

      {/* Detail Modal */}
      {selectedApp && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4" onClick={() => setSelectedApp(null)}>
          <Card className="max-w-3xl w-full max-h-[90vh] overflow-y-auto" onClick={(e) => e.stopPropagation()}>
            <div className="p-8">
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-2xl font-bold">Başvuru Detayları</h2>
                <Button variant="outline" onClick={() => setSelectedApp(null)}>
                  <XCircle className="h-4 w-4" />
                </Button>
              </div>

              <div className="space-y-6">
                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="text-sm font-medium text-gray-600">Başvuru No</label>
                    <p className="font-mono text-lg">#{selectedApp.id}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Başvuru Tarihi</label>
                    <p className="text-lg">{selectedApp.date}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Ad Soyad</label>
                    <p className="text-lg">{selectedApp.firstName} {selectedApp.lastName || selectedApp.name}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">E-posta</label>
                    <p className="text-lg">{selectedApp.email}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Telefon</label>
                    <p className="text-lg">{selectedApp.phone}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Tutar</label>
                    <p className="text-lg font-bold text-emerald-600">{selectedApp.amount}</p>
                  </div>
                </div>

                {selectedApp.type === "Vize" && (
                  <div className="border-t pt-6">
                    <h3 className="font-bold text-lg mb-4">Vize Bilgileri</h3>
                    <div className="grid md:grid-cols-2 gap-4">
                      <div>
                        <label className="text-sm font-medium text-gray-600">Ülke</label>
                        <p>{selectedApp.country}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-gray-600">Vize Türü</label>
                        <p>{selectedApp.visaType}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-gray-600">Pasaport No</label>
                        <p>{selectedApp.passportNo}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-gray-600">Seyahat Tarihi</label>
                        <p>{selectedApp.travelDate}</p>
                      </div>
                    </div>
                  </div>
                )}

                {selectedApp.type === "Dil Okulu" && (
                  <div className="border-t pt-6">
                    <h3 className="font-bold text-lg mb-4">Okul Bilgileri</h3>
                    <div className="grid md:grid-cols-2 gap-4">
                      <div>
                        <label className="text-sm font-medium text-gray-600">Okul</label>
                        <p>{selectedApp.school}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-gray-600">Kurs</label>
                        <p>{selectedApp.course}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-gray-600">Başlangıç</label>
                        <p>{selectedApp.startDate}</p>
                      </div>
                      <div>
                        <label className="text-sm font-medium text-gray-600">Süre</label>
                        <p>{selectedApp.duration}</p>
                      </div>
                    </div>
                  </div>
                )}

                {selectedApp.documents && (
                  <div className="border-t pt-6">
                    <h3 className="font-bold text-lg mb-4">Yüklenen Evraklar</h3>
                    <div className="space-y-2">
                      {selectedApp.documents.map((doc: any, idx: number) => (
                        <div key={idx} className="flex items-center justify-between p-3 bg-gray-50 rounded">
                          <div className="flex items-center gap-2">
                            <FileText className="h-4 w-4 text-emerald-600" />
                            <span className="text-sm">{doc.name || doc}</span>
                          </div>
                          <CheckCircle2 className="h-4 w-4 text-green-600" />
                        </div>
                      ))}
                    </div>
                  </div>
                )}

                <div className="border-t pt-6 flex gap-3">
                  <Button
                    className="flex-1 bg-green-600 hover:bg-green-700"
                    onClick={() => {
                      updateStatus(selectedApp.id, "Onaylandı", selectedApp.type === "Vize" ? "application" : "enrollment");
                      setSelectedApp(null);
                    }}
                  >
                    <CheckCircle2 className="h-4 w-4 mr-2" />
                    Onayla
                  </Button>
                  <Button
                    variant="destructive"
                    className="flex-1"
                    onClick={() => {
                      updateStatus(selectedApp.id, "Reddedildi", selectedApp.type === "Vize" ? "application" : "enrollment");
                      setSelectedApp(null);
                    }}
                  >
                    <XCircle className="h-4 w-4 mr-2" />
                    Reddet
                  </Button>
                </div>
              </div>
            </div>
          </Card>
        </div>
      )}
    </div>
  );
}
